<?php

namespace App\Models\Auth\Traits\Attribute;

use Illuminate\Support\Facades\Hash;

/**
 * Trait UserAttribute.
 */
trait UserAttribute
{
    /**
     * @param $password
     */
    public function setPasswordAttribute($password) : void
    {
        // If password was accidentally passed in already hashed, try not to double hash it
        if (
            (\strlen($password) === 60 && preg_match('/^\$2y\$/', $password)) ||
            (\strlen($password) === 95 && preg_match('/^\$argon2i\$/', $password))
        ) {
            $hash = $password;
        } else {
            $hash = Hash::make($password);
        }

        // Note: Password Histories are logged from the \App\Observer\User\UserObserver class
        $this->attributes['password'] = $hash;
    }

    /**
     * @return string
     */
    public function getStatusLabelAttribute()
    {
        if ($this->isActive()) {
            return "<span class='badge badge-success'>".__('labels.general.active').'</span>';
        }

        return "<span class='badge badge-danger'>".__('labels.general.inactive').'</span>';
    }

    /**
     * @return string
     */
    public function getConfirmedLabelAttribute()
    {
        if ($this->isConfirmed()) {
            if ($this->id != 1 && $this->id != auth()->id()) {
                return '<a href="'.route(
                    'admin.auth.user.unconfirm',
                        $this
                ).'" data-toggle="tooltip" data-placement="top" title="'.__('buttons.backend.access.users.unconfirm').'" name="confirm_item"><span class="pill-publish" style="cursor:pointer">'.__('labels.general.yes').'</span></a>';
            } else {
                return '<span class="pill-publish">'.__('labels.general.yes').'</span>';
            }
        }

        return '<a href="'.route('admin.auth.user.confirm', $this).'" data-toggle="tooltip" data-placement="top" title="'.__('buttons.backend.access.users.confirm').'" name="confirm_item"><span class="badge badge-danger" style="cursor:pointer">'.__('labels.general.no').'</span></a>';
    }

    /**
     * @return string
     */
    public function getRolesLabelAttribute()
    {
        $roles = $this->getRoleNames()->toArray();

        if (\count($roles)) {
            return implode(', ', array_map(function ($item) {
                return ucwords($item);
            }, $roles));
        }

        return 'N/A';
    }

    /**
     * @return string
     */
    public function getPermissionsLabelAttribute()
    {
        $permissions = $this->getDirectPermissions()->toArray();

        if (\count($permissions)) {
            return implode(', ', array_map(function ($item) {
                return ucwords($item['name']);
            }, $permissions));
        }

        return 'N/A';
    }

    /**
     * @return string
     */
    public function getFullNameAttribute()
    {
        return $this->last_name
            ? $this->first_name.' '.$this->last_name
            : $this->first_name;
    }

    /**
     * @return string
     */
    public function getNameAttribute()
    {
        return $this->full_name;
    }

    /**
     * @return mixed
     */
    public function getPictureAttribute()
    {
        return $this->getPicture();
    }

    /**
     * @return string
     */
    public function getSocialButtonsAttribute()
    {
        $accounts = [];

        foreach ($this->providers as $social) {
            $accounts[] = '<a href="'.route(
                'admin.auth.user.social.unlink',
                    [$this, $social]
            ).'" data-toggle="tooltip" data-placement="top" title="'.__('buttons.backend.access.users.unlink').'" data-method="delete"><i class="fab fa-'.$social->provider.'"></i></a>';
        }

        return \count($accounts) ? implode(' ', $accounts) : __('labels.general.none');
    }

    /**
     * @return string
     */
    public function getLoginAsButtonAttribute()
    {
        /*
         * If the admin is currently NOT spoofing a user
         */
        if (! session()->has('admin_user_id') || ! session()->has('temp_user_id')) {
            //Won't break, but don't let them "Login As" themselves
            if ($this->id != auth()->id()) {
                return '<a href="'.route(
                    'admin.auth.user.login-as',
                        $this
                ).'" class="add-btn mt-2"><i class="fas fa-sign-in-alt mr-2"></i>'.__('buttons.backend.access.users.login_as', ['user' => e($this->full_name)]).'</a> ';
            }
        }

        return '';
    }

    /**
     * @return string
     */
    public function getClearSessionButtonAttribute()
    {
        if ($this->id != auth()->id() && config('session.driver') == 'database') {
            return '<a href="'.route('admin.auth.user.clear-session', $this).'"
			 	 data-trans-button-cancel="'.__('buttons.general.cancel').'"
                 data-trans-button-confirm="'.__('buttons.general.continue').'"
                 data-trans-title="'.__('strings.backend.general.are_you_sure').'"
                 class="dropdown-item" name="confirm_item">'.__('buttons.backend.access.users.clear_session').'</a> ';
        }

        return '';
    }

    /**
     * @return string
     */
    public function getShowButtonAttribute()
    {
        return '<a href="'.route('admin.auth.user.show', $this).'" data-toggle="tooltip" data-placement="top" title="'.__('buttons.general.crud.view').'" class="btn btn-info"><i class="fas fa-eye"></i></a>';
    }

    /**
     * @return string
     */
    public function getEditButtonAttribute()
    {
        return '<a href="'.route('admin.auth.user.edit', $this).'" data-toggle="tooltip" data-placement="top" title="'.__('buttons.general.crud.edit').'" class="btn btn-primary"><i class="fas fa-edit"></i></a>';
    }

    /**
     * @return string
     */
    public function getChangePasswordButtonAttribute()
    {
        return '<a title="Change Password" href="'.route('admin.auth.user.change-password', $this).'" class=""><i class="fa fa-unlock-alt" aria-hidden="true"></i></a> ';
    }

    /**
     * @return string
     */
    public function getStatusButtonAttribute()
    {
        if ($this->id != auth()->id()) {
            switch ($this->active) {
                case 0:
                    return '<a title="Activate" href="'.route('admin.auth.user.mark', [
                            $this,
                            1,
                        ]).'" class=""><i class="fas fa-check-circle"></i></a> ';

                case 1:
                    return '<a title="Deactivate" href="'.route('admin.auth.user.mark', [
                            $this,
                            0,
                        ]).'" class=""><i class="fas fa-ban"></i></a> ';

                default:
                    return '';
            }
        }

        return '';
    }

    /**
     * @return string
     */
    public function getConfirmedButtonAttribute()
    {
        if (! $this->isConfirmed() && ! config('access.users.requires_approval')) {
            return '<a href="'.route('admin.auth.user.account.confirm.resend', $this).'" class="dropdown-item">'.__('buttons.backend.access.users.resend_email').'</a> ';
        }

        return '';
    }

    /**
     * @return string
     */
    public function getDeleteButtonAttribute()
    {

        if ($this->id != auth()->id() && $this->id != 1) {

           return    view('backend.datatable.action-delete')
               ->with(['route' => route('admin.auth.user.destroy', $this),'class' => 'dropdown-item'] )
               ->render();
        }

        return '';
    }

    /**
     * @return string
     */
    public function getDeletePermanentlyButtonAttribute()
    {
        return '<a href="'.route('admin.auth.user.delete-permanently', $this).'" name="confirm_item" class="btn btn-danger"><i class="fas fa-trash" data-toggle="tooltip" data-placement="top" title="'.__('buttons.backend.access.users.delete_permanently').'"></i></a> ';
    }

    /**
     * @return string
     */
    public function getRestoreButtonAttribute()
    {
        return '<a href="'.route('admin.auth.user.restore', $this).'" name="confirm_item" class="btn btn-info"><i class="fas fa-sync" data-toggle="tooltip" data-placement="top" title="'.__('buttons.backend.access.users.restore_user').'"></i></a> ';
    }

    /**
     * @return string
     */
    // public function getActionButtonsAttribute()
    // {
    //     if ($this->trashed()) {
    //         return '
	// 			<div class="btn-group" role="group" aria-label="'.__('labels.backend.access.users.user_actions').'">
	// 			  '.$this->restore_button.'
	// 			  '.$this->delete_permanently_button.'
	// 			</div>';
    //     }

    //     return '
    // 	<div class="btn-group" role="group" aria-label="'.__('labels.backend.access.users.user_actions').'">
	// 	  '.$this->show_button.'
	// 	  '.$this->edit_button.'

	// 	  <div class="btn-group btn-group-sm" role="group">
	// 		<button id="userActions" type="button" class="btn btn-secondary dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
	// 		  '.__('labels.general.more').'
	// 		</button>
	// 		<div class="dropdown-menu" aria-labelledby="userActions">
	// 		  '.$this->clear_session_button.'
	// 		  '.$this->login_as_button.'
	// 		  '.$this->change_password_button.'
	// 		  '.$this->status_button.'
	// 		  '.$this->confirmed_button.'
	// 		</div>
	// 	  </div>
	// 	</div>';
    // }
    public function getActionButtonsAttribute()
{
    // For trashed users (soft-deleted)
    if ($this->trashed()) {
        return '
            <div class="btn-group" role="group" aria-label="'.__('labels.backend.access.users.user_actions').'">
                '.$this->restore_button.'
                '.$this->delete_permanently_button.'
            </div>';
    }

    // For active users — all actions in ellipsis dropdown
    return '
    <div class="text-center">
    <div class="action-pill">
        
        
            <a class="" href="'.route('admin.auth.user.show', $this->id).'">
                 <i class="fas fa-eye"></i>
            </a>
            <a class="" href="'.route('admin.auth.user.edit', $this->id).'">
                 <i class="fas fa-edit"></i>
            </a>
           
            '.$this->clear_session_button.'
            
            '.$this->change_password_button.'
            '.$this->status_button.'
            '.$this->confirmed_button.'
        </div>
        <div class="mt-2 mb-2">
        '.$this->login_as_button.'
        </div>
    </div>';
}
}
